// Loads site-config.json and updates the page with site-wide settings
// Add cache-busting query parameter to ensure fresh config is loaded

// Create a promise that resolves when config is fully loaded
window.siteConfigLoaded = new Promise((resolve, reject) => {
  fetch('/config/site-config.json?t=' + Date.now())
    .then(response => response.json())
    .then(config => {
      // Preserve any page-specific title and append site title.
      // Examples:
      //  - If page has <title>Home</title> -> becomes "Home | Site Title"
      //  - If page has "Home | Old Site" -> becomes "Home | Site Title"
      const originalTitle = (document.title || '').trim();
      const path = window.location.pathname || '';
      const isHomePath = path === '/' || path.endsWith('/index.html') || path === '/index.html';

      // If we're on the home page, prefer a standard 'Home' page title to be
      // used as the left-hand side (so pages render like "Home | Site Title").
      if (isHomePath) {
        document.title = (config.title ? 'Home' + ' | ' + config.title : 'Home');
      } else if (!originalTitle) {
        document.title = config.title || '';
      } else if (originalTitle.includes('|')) {
        // Replace right-hand side with site title while keeping left (page) title
        const parts = originalTitle.split('|').map(s => s.trim());
        const pageTitle = parts[0] || '';
        document.title = pageTitle + (config.title ? ' | ' + config.title : (parts.slice(1).join(' | ') || ''));
      } else if (originalTitle === config.title) {
        document.title = config.title;
      } else {
        document.title = originalTitle + (config.title ? ' | ' + config.title : '');
      }

      // Set tagline (preserve inner HTML with icon)
      document.querySelectorAll('.site-tagline').forEach(el => {
        // Find and update only the text node after the icon
        const icon = el.querySelector('i');
        if (icon) {
          // Keep the icon, update text after it
          el.innerHTML = icon.outerHTML + ' ' + (config.tagline || el.textContent.replace(icon.outerHTML, '').trim());
        } else {
          el.textContent = config.tagline || el.textContent;
        }
      });

      // Set logo lines
      document.querySelectorAll('.site-logo-line1').forEach(el => {
        el.textContent = config.logoLine1 || el.textContent;
      });
      document.querySelectorAll('.site-logo-line2').forEach(el => {
        el.textContent = config.logoLine2 || el.textContent;
      });

      // Utility to set link text (handles anchor that contains a span)
      function setLinkTextAndHref(el, text, href) {
        if (!el) return;
        if (el.tagName === 'A') {
          el.href = href;
          const span = el.querySelector('span');
          if (span) span.textContent = text;
          else el.textContent = text;
        } else {
          // Non-anchor (like a <p>), set text only
          el.textContent = text;
        }
      }

      // Update all email links and text
      document.querySelectorAll('.site-email').forEach(el => {
        setLinkTextAndHref(el, config.email, 'mailto:' + config.email);
      });

      // Update all phone links and text
      const telHref = 'tel:' + (config.phone ? config.phone.replace(/[^\d]/g, '') : '');
      document.querySelectorAll('.site-phone').forEach(el => {
        setLinkTextAndHref(el, config.phone, telHref);
      });

      // Inject Chatway livechat widget when a chatwayId is provided in config
      // The site stores only the ID (e.g. "mqKY4PL5jf1N"). The full src is
      // https://cdn.chatway.app/widget.js?id=<ID>
      try {
        if (config.chatwayId) {
          const base = 'https://cdn.chatway.app/widget.js?id=';
          const id = String(config.chatwayId).trim();
          if (id) {
            const existing = document.getElementById('chatway');
            const src = base + encodeURIComponent(id);
            console.log('[site-config] Chatway ID present, attempting injection:', id);
            if (existing) {
              // Update src if it differs
              if (existing.src.indexOf(id) === -1) {
                console.log('[site-config] Updating existing #chatway src to', src);
                existing.src = src;
              } else {
                console.log('[site-config] Chatway script already present with same ID');
              }
            } else {
              const s = document.createElement('script');
              s.id = 'chatway';
              s.async = true;
              s.src = src;
              document.head.appendChild(s);
              console.log('[site-config] Appended Chatway script to head:', src);
            }
          }
        }
      } catch (e) {
        console.warn('Chatway injection failed', e);
      }

      // Resolve the promise when config is fully loaded
      resolve(config);
    })
    .catch(error => {
      console.error('Failed to load site config:', error);
      reject(error);
    });
});
